package net.minecraft.src.ic2.advancedmachines;

import java.util.*;
import net.minecraft.src.*;
import net.minecraft.src.forge.*;
import net.minecraft.src.ic2.api.*;
import net.minecraft.src.ic2.api.Direction;

public abstract class TileEntityAdvancedMachine extends TileEntityBaseMachine implements ISidedInventory
{
    private static final int MAX_PROGRESS = 4000;
    private static final int MAX_ENERGY = 5000;
    private static final int MAX_SPEED = 7500;
    private static final int MAX_INPUT = 32;
    private String inventoryName;
    private int[] inputs;
    private int[] outputs;
    short speed;
    short progress;
    private String dataFormat;
    private int dataScaling;
    
    private IC2AudioSource audioSource;
    private static final int EventStart = 0;
    private static final int EventInterrupt = 1;
    private static final int EventStop = 2;
    
    private int energyConsume = 2;
    private int defaultEnergyConsume = 3;
    private int acceleration = 1;
    private int defaultAcceleration = 1;
    private int maxSpeed;
    
    private ItemStack[] lastInventory;

    public TileEntityAdvancedMachine(String var1, String var2, int var3, int[] var4, int[] var5)
    {
        super(var4.length + var5.length + 5, var4.length, MAX_ENERGY, MAX_INPUT);
        this.inventoryName = var1;
        this.dataFormat = var2;
        this.dataScaling = var3;
        this.inputs = var4;
        this.outputs = var5;
        this.speed = 0;
        this.progress = 0;
    }

    public void readFromNBT(NBTTagCompound var1)
    {
        super.readFromNBT(var1);
        this.speed = var1.getShort("speed");
        this.progress = var1.getShort("progress");
    }

    public void writeToNBT(NBTTagCompound var1)
    {
        super.writeToNBT(var1);
        var1.setShort("speed", this.speed);
        var1.setShort("progress", this.progress);
    }

    public String getInvName()
    {
        return this.inventoryName;
    }

    public int gaugeProgressScaled(int var1)
    {
        return var1 * this.progress / MAX_PROGRESS;
    }

    public int gaugeFuelScaled(int var1)
    {
        return var1 * this.energy / this.maxEnergy;
    }

    public void updateEntity()
    {
        super.updateEntity();
        boolean newItemProcessing = false;
        if (this.energy <= this.maxEnergy)
        {
            newItemProcessing = this.provideEnergy();
        }

        boolean isActive = this.getActive();
        if (this.progress >= MAX_PROGRESS)
        {
            this.operate();
            newItemProcessing = true;
            this.progress = 0;
            isActive = false;
            
            NetworkHelper.initiateTileEntityEvent(this, EventStop, true);
        }

        boolean bCanOperate = this.canOperate();
        if (this.energy > 0 && (bCanOperate || this.isRedstonePowered()))
        {
        	setOverclockRates();
        	
            if (this.speed < maxSpeed)
            {
                this.speed += acceleration;
                this.energy -= energyConsume;
            }
            else
            {
            	this.speed = (short) maxSpeed;
            	this.energy -= defaultEnergyConsume;
            }

            isActive = true;
            NetworkHelper.initiateTileEntityEvent(this, EventStart, true);
        }
        else
        {
        	boolean wasWorking = this.speed != 0;
            this.speed = (short)(this.speed - Math.min(this.speed, 4));
            if (wasWorking && this.speed == 0)
            {
            	NetworkHelper.initiateTileEntityEvent(this, EventInterrupt, true);
            }
        }

        if (isActive && this.progress != 0)
        {
            if (!bCanOperate || this.speed == 0)
            {
                if (!bCanOperate)
                {
                    this.progress = 0;
                }

                isActive = false;
            }
        }
        else if (bCanOperate)
        {
            if (this.speed != 0)
            {
                isActive = true;
            }
        }
        else
        {
            this.progress = 0;
        }

        if (isActive && bCanOperate)
        {
            this.progress = (short)(this.progress + this.speed / 30);
        }

        if (newItemProcessing)
        {
            this.onInventoryChanged();
        }
        /*
        if(isActive != getActive())
    	{
    		worldObj.markBlockNeedsUpdate(xCoord, yCoord, zCoord);
    		setActive(isActive);
    	}
    	*/
    }
    
    public int injectEnergy(Direction var1, int var2)
    {
    	this.setOverclockRates();
		return super.injectEnergy(var1, var2);
    }

    public void operate()
    {    	
        if (this.canOperate())
        {
            ItemStack resultStack = this.getResultFor(this.inventory[this.inputs[0]], false).copy();
            int[] var2 = new int[this.outputs.length];
            int var3 = resultStack.getMaxStackSize();

            int var4;
            for (var4 = 0; var4 < this.outputs.length; ++var4)
            {
                if (this.inventory[this.outputs[var4]] == null)
                {
                    var2[var4] = var3;
                }
                else if (this.inventory[this.outputs[var4]].isItemEqual(resultStack))
                {
                    var2[var4] = var3 - this.inventory[this.outputs[var4]].stackSize;
                }
            }

            for (var4 = 0; var4 < var2.length; ++var4)
            {
                if (var2[var4] > 0)
                {
                    int var5 = Math.min(resultStack.stackSize, var2[var4]);
                    if (this.inventory[this.outputs[var4]] == null)
                    {
                        this.inventory[this.outputs[var4]] = resultStack.copy();
                        onFinishedProcessingItem();
                    }
                    else
                    {
                        this.inventory[this.outputs[var4]].stackSize += var5;
                        onFinishedProcessingItem();
                    }

                    resultStack.stackSize -= var5;
                }

                if (resultStack.stackSize <= 0)
                {
                    break;
                }
            }

            if (this.inventory[this.inputs[0]].getItem().hasContainerItem())
            {
                this.inventory[this.inputs[0]] = new ItemStack(this.inventory[this.inputs[0]].getItem().getContainerItem());
            }
            else
            {
                --this.inventory[this.inputs[0]].stackSize;
            }

            if (this.inventory[this.inputs[0]].stackSize <= 0)
            {
                this.inventory[this.inputs[0]] = null;
            }
        }
    }
    
    public void onFinishedProcessingItem()
    {
    	
    }

    public boolean canOperate()
    {
        if (this.inventory[this.inputs[0]] == null)
        {
            return false;
        }
        else
        {
            ItemStack var1 = this.getResultFor(this.inventory[this.inputs[0]], false);
            if (var1 == null)
            {
                return false;
            }
            else
            {
                int var2 = var1.getMaxStackSize();
                int var3 = 0;
                int[] var4 = this.outputs;
                int var5 = var4.length;

                for (int var6 = 0; var6 < var5; ++var6)
                {
                    int var7 = var4[var6];
                    if (this.inventory[var7] == null)
                    {
                        var3 += var2;
                    }
                    else if (this.inventory[var7].isItemEqual(var1))
                    {
                        var3 += var2 - this.inventory[var7].stackSize;
                    }
                }

                return var3 >= var1.stackSize;
            }
        }
    }

    public abstract ItemStack getResultFor(ItemStack var1, boolean var2);

    protected abstract List getResultMap();

    public abstract Container getGuiContainer(InventoryPlayer var1);

    @Override
	public
    int getStartInventorySide(int side)
    {
		switch (side)
		{
			case 0: return 1; // 0 bottom, power slot, 1
			case 1: return 0; // 1 top side, input, 0
			default: return 2; // anything other is sides eg output, starts at slot 2
		}
    }

    public int getSizeInventorySide(int var1)
    {
        switch (var1)
        {
            case 0:
                return 1;
            case 1:
                return this.inputs.length;
            default:
                return this.outputs.length;
        }
    }

    public String printFormattedData()
    {
        return String.format(this.dataFormat, new Object[] {Integer.valueOf(this.speed * this.dataScaling)});
    }
    
    public void invalidate()
    {
    	if (this.audioSource != null)
    	{
    		IC2AudioSource.removeSource(audioSource);
    		this.audioSource = null;
    	}
    	super.invalidate();
    }
    
    public String getStartSoundFile()
    {
    	return null;
    }

    public String getInterruptSoundFile()
    {
    	return null;
    }

    @Override
    public void onNetworkEvent(int event)
    {
    	super.onNetworkEvent(event);
    	
    	if ((this.audioSource == null) && (getStartSoundFile() != null))
    	{
    		this.audioSource = new IC2AudioSource(this, getStartSoundFile());
    	}

    	switch (event)
    	{
	    	case EventStart:
	    		this.setActiveWithoutNotify(true);
	    		if (this.audioSource == null) break;
	    		this.audioSource.play();
	    		break;
	    	case EventInterrupt:
	    		this.setActiveWithoutNotify(false);
	    		if (this.audioSource == null) break;
	    		this.audioSource.stop();
	    		if (getInterruptSoundFile() == null) break;
	    		IC2AudioSource.playOnce(this, getInterruptSoundFile());
	    		break;
	    	case EventStop:
	    		this.setActiveWithoutNotify(false);
	    		if (this.audioSource == null) break;
	    		this.audioSource.stop();
    	}
    	
    	NetworkHelper.announceBlockUpdate(worldObj, xCoord, yCoord, zCoord);
    }
    
    public int getUpgradeSlotsStartSlot()
    {
    	return inventory.length-4;
    }
    
    public void setOverclockRates()
    {
    	if (lastInventory != null && lastInventory.equals(this.inventory)) return;
    	lastInventory = inventory.clone();
    	
    	int overclockerUpgradeCount = 0;
    	int transformerUpgradeCount = 0;
    	int energyStorageUpgradeCount = 0;

    	for (int i = 0; i < 4; i++) {
    		ItemStack itemStack = this.inventory[getUpgradeSlotsStartSlot() + i];

    		if (itemStack != null) {
    			if (itemStack.isItemEqual(mod_IC2AdvancedMachines.overClockerStack))
    				overclockerUpgradeCount += itemStack.stackSize;
    			else if (itemStack.isItemEqual(mod_IC2AdvancedMachines.transformerStack))
    				transformerUpgradeCount += itemStack.stackSize;
    			else if (itemStack.isItemEqual(mod_IC2AdvancedMachines.energyStorageUpgradeStack)) {
    				energyStorageUpgradeCount += itemStack.stackSize;
    			}
    		}
    	}

    	if (overclockerUpgradeCount > 32) overclockerUpgradeCount = 32;
    	if (transformerUpgradeCount > 10) transformerUpgradeCount = 10;

    	this.energyConsume = (int)(this.defaultEnergyConsume * Math.pow(2.0D, overclockerUpgradeCount));
    	this.acceleration = (int)(((this.defaultAcceleration*2) * Math.pow(1.6D, overclockerUpgradeCount)) /2);
    	this.maxSpeed = (MAX_SPEED + overclockerUpgradeCount * 500);
    	this.maxInput = (MAX_INPUT * (int)Math.pow(4.0D, transformerUpgradeCount));
    	this.maxEnergy = (MAX_ENERGY + energyStorageUpgradeCount * MAX_ENERGY + this.maxInput - 1);
    	this.tier = (transformerUpgradeCount);
    }
}
