package immibis.chunkloader;

import java.util.*;

import net.minecraft.server.*;

public class WorldInfo extends WorldMapBase {
	
	private World world;
	
	public static class XYZ {
		int x, y, z;
		public XYZ(int x, int y, int z) {
			this.x = x;
			this.y = y;
			this.z = z;
		}
		public int hashCode() {
			return x * 23434 + y * 2351321 + z;
		}
		public boolean equals(Object o) {
			try {
				XYZ _o = (XYZ)o;
				return x == _o.x && y == _o.y && z == _o.z;
			} catch(ClassCastException e) {
				return false;
			}
		}
		public XYZ(TileEntity tile) {
			this(tile.x, tile.y, tile.z);
		}
		
		public String toString() {
			return "["+x+","+y+","+z+"]";
		}
	}
	
	public static class LoaderInfo {
		public LoaderInfo(XYZ pos, WorldInfo world, String player, int radius) {
			this.pos = pos;
			this.removeTime = -1;
			this.player = player;
			this.radius = radius;
			this.world = world;
		}
		
		public LoaderInfo(NBTTagCompound tag) {
			this.pos = new XYZ(tag.getInt("X"), tag.getInt("Y"), tag.getInt("Z"));
			this.removeTime = tag.getLong("removeTime");
			this.player = tag.getString("player");
			if(this.player.equals(""))
				this.player = null;
			this.isServerOwned = tag.getBoolean("serverOwned");
		}
		
		public String toString() {
			return "(" + pos + ", " + player + ", r=" + radius + ")";
		}
		
		public long removeTime; // -1 if not waiting to be removed
		public XYZ pos;
		public String player;
		public int radius;
		public WorldInfo world;
		public boolean isServerOwned;
		
		public NBTTagCompound writeNBT() {
			NBTTagCompound tag = new NBTTagCompound();
			tag.setInt("X", pos.x);
			tag.setInt("Y", pos.y);
			tag.setInt("Z", pos.z);
			tag.setLong("removeTime", removeTime);
			tag.setString("player", player == null ? "" : player);
			tag.setBoolean("serverOwned", isServerOwned);
			return tag;
		}

		public Collection<ChunkCoordIntPair> getLoadedChunks() {
			if(radius < 0 || player == null)
				return Collections.emptyList();
			
			int cx = pos.x >> 4;
			int cz = pos.z >> 4;
			int r2 = radius*2+1;
			ChunkCoordIntPair[] a = new ChunkCoordIntPair[r2 * r2];
			int pos = 0;
			for(int dx = -radius; dx <= radius; dx++) {
				for(int dz = -radius; dz <= radius; dz++) {
					a[pos++] = new ChunkCoordIntPair(cx + dx, cz + dz);
				}
			}
			return Arrays.asList(a);
		}

		public String getLogString() {
			return "owner="+player+", radius="+radius;
		}
	}
	
	private long checkTime = 0;
	
	// stores the number of chunk loaders keeping a chunk loaded
	private HashMap<ChunkCoordIntPair, Integer> loadedChunks = new HashMap<ChunkCoordIntPair, Integer>();
	
	private HashMap<String, Integer> curQuota = new HashMap<String, Integer>();
	
	public void addLoadedChunk(ChunkCoordIntPair ccip, String player) {
		Integer val = loadedChunks.get(ccip);
		if(val != null)
			loadedChunks.put(ccip, val + 1);
		else
			loadedChunks.put(ccip, 1);
		
		if(player != null) {
			val = curQuota.get(player);
			if(val != null)
				curQuota.put(player, val + 1);
			else
				curQuota.put(player, 1);
		}
		
		if(Main.DEBUG)
			System.out.println("addLoadedChunk("+ccip+","+player+")");
	}
	public void removeLoadedChunk(ChunkCoordIntPair ccip, String player) {
		Integer val = loadedChunks.get(ccip);
		if(val != null) {
			if(val.intValue() == 1)
				loadedChunks.remove(ccip);
			else
				loadedChunks.put(ccip, val - 1);
		}
		
		if(player != null) {
			val = curQuota.get(player);
			if(val != null) {
				if(val.intValue() == 1)
					curQuota.remove(player);
				else
					curQuota.put(player, val - 1);
			}
		}
		
		if(Main.DEBUG)
			System.out.println("removeLoadedChunk("+ccip+","+player+")");
	}
	
	private HashMap<XYZ, LoaderInfo> loaders = new HashMap<XYZ, LoaderInfo>();
	private ArrayList<LoaderInfo> toRemove = new ArrayList<LoaderInfo>();
	
	public WorldInfo(String name) {
		super(name);
	}
	
	private static WorldInfo loadFrom(World w, String mapname) {
		return (WorldInfo)w.worldMaps.get(WorldInfo.class, mapname);
	}

	public static WorldInfo get(World w) {
		String mapname = "ICL-" + w.getWorld().getName();
		WorldInfo wi = loadFrom(w, mapname);
		if(wi != null) {
			wi.checkTime = w.getTime() + 40;
			wi.world = w;
			return wi;
		}
		
		wi = new WorldInfo(mapname);
		wi.world = w;
		w.worldMaps.a(mapname, wi);
		return wi;
	}

	@Override
	public void a(NBTTagCompound var1) {
		loaders.clear();
		toRemove.clear();
		{
			NBTTagList list = var1.getList("loaders");
			for(int k = 0; k < list.size(); k++) {
				NBTTagCompound c = (NBTTagCompound)list.get(k);
				LoaderInfo loader = new LoaderInfo(c);
				loaders.put(loader.pos, loader);
				if(loader.removeTime != -1)
					toRemove.add(loader);
				for(ChunkCoordIntPair ccip : loader.getLoadedChunks())
					addLoadedChunk(ccip, loader.isServerOwned ? null : loader.player);
			}
		}
	}

	@Override
	public void b(NBTTagCompound var1) {
		{
			NBTTagList list = new NBTTagList();
			for(LoaderInfo l : loaders.values())
				list.add(l.writeNBT());
			var1.set("loaders", list);
		}
	}
	
	public void setDirty(boolean val) {
		a(val);
	}

	public void delayRemoveLoader(TileChunkLoader tile) {
		LoaderInfo loader = loaders.get(new XYZ(tile));
		loader.removeTime = world.getTime() + 20; // remove it in one second
		toRemove.add(loader);
		if(Main.DEBUG)
			System.out.println("Removing "+tile.x+","+tile.y+","+tile.z+" in one second");
		setDirty(true);
	}

	public void addLoader(TileChunkLoader tile) {
		XYZ pos = new XYZ(tile);
		{
			LoaderInfo loader = loaders.get(pos);
			if(loader != null)
				removeLoader(loader);
		}
		LoaderInfo l = tile.getLoaderInfo();
		if(Main.DEBUG)
			System.out.println("addLoader(" + l + ")");
		loaders.put(pos, l);
		for(ChunkCoordIntPair ccip : l.getLoadedChunks())
			addLoadedChunk(ccip, l.isServerOwned ? null : l.player);
		setDirty(true);
	}
	
	private void removeLoader(LoaderInfo loader) {
		if(Main.DEBUG)
			System.out.println("removeLoader(" + loader + ")");
		loaders.remove(loader.pos);
		toRemove.remove(loader);
		for(ChunkCoordIntPair ccip : loader.getLoadedChunks())
			removeLoadedChunk(ccip, loader.isServerOwned ? null : loader.player);
		setDirty(true);
	}

	public void tick() {
		if(toRemove.size() > 0) {
			LoaderInfo check = toRemove.get(world.random.nextInt(toRemove.size()));
			if(check.removeTime < world.getTime()) {
				removeLoader(check);
			}
		}
		// Rebuild the loaded chunks and loaders list a short time after loading a world
		if(checkTime != -1 && checkTime < world.getTime()) {
			LinkedList<LoaderInfo> copy = new LinkedList<LoaderInfo>(loaders.values());
			loaders.clear();
			loadedChunks.clear();
			curQuota.clear();
			checkTime = -1;
			for(LoaderInfo li : copy)
			{
				TileEntity te = world.getTileEntity(li.pos.x, li.pos.y, li.pos.z);
				if(te instanceof TileChunkLoader)
					addLoader((TileChunkLoader)te);
			}
		}
	}
	
	public Collection<? extends ChunkCoordIntPair> getLoadedChunks() {
		return loadedChunks.keySet();
	}
	
	public boolean isChunkForceLoaded(ChunkCoordIntPair pos) {
		return loadedChunks.get(pos) != null;
	}
	
	public int getCurQuota(String player) {
		Integer val = curQuota.get(player);
		return val == null ? 0 : val;
	}
	public void readdLoader(TileChunkLoader tile) {
		removeLoader(loaders.get(new XYZ(tile)));
		addLoader(tile);
	}
	public Collection<LoaderInfo> getAllLoaders() {
		return loaders.values();
	}
	
	public String getName() {
		return world.getWorld().getName();
	}
}
